<?php

namespace App\Models;

use App\Library\SmsBuilder;
use App\Library\Tool;
use Aws\Sns\Exception\SnsException;
use Aws\Sns\SnsClient;
use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Translation\Translator;
use Illuminate\Database\Eloquent\Model;
use Plivo\Exceptions\PlivoResponseException;
use Plivo\RestClient;
use Psr\Http\Client\ClientExceptionInterface;
use smpp\SMPP;

//use SMSGatewayMe\Client\Api\MessageApi;
//use SMSGatewayMe\Client\ApiClient;
//use SMSGatewayMe\Client\ApiException;
//use SMSGatewayMe\Client\Configuration;
use Twilio\Exceptions\ConfigurationException;
use Twilio\Exceptions\TwilioException;
use Twilio\Rest\Client;
use Vonage\Client\Credentials\Basic;
use Vonage\SMS\Message\SMS;

class SendCampaignSMS extends Model
{

    /**
     * make normal message to unicode message
     *
     * @param $message
     *
     * @return string
     */
    private function sms_unicode($message): string
    {
        $hex1 = '';
        if (function_exists('iconv')) {
            $latin = @iconv('UTF−8', 'ISO−8859−1', $message);
            if (strcmp($latin, $message)) {
                $arr  = unpack('H*hex', @iconv('UTF-8', 'UCS-2BE', $message));
                $hex1 = strtoupper($arr['hex']);
            }
            if ($hex1 == '') {
                $hex2 = '';
                $hex  = '';
                for ($i = 0; $i < strlen($message); $i++) {
                    $hex = dechex(ord($message[$i]));
                    $len = strlen($hex);
                    $add = 4 - $len;
                    if ($len < 4) {
                        for ($j = 0; $j < $add; $j++) {
                            $hex = "0".$hex;
                        }
                    }
                    $hex2 .= $hex;
                }

                return $hex2;
            } else {
                return $hex1;
            }
        } else {
            return 'failed';
        }
    }


    /**
     *
     * send plain message
     *
     * @param $data
     *
     * @return array|Application|Translator|string|null
     */
    public function sendPlainSMS($data)
    {
        $phone          = $data['phone'];
        $sending_server = $data['sending_server'];
        $gateway_name   = $data['sending_server']->settings;
        $message        = null;
        $sms_type       = $data['sms_type'];
        $get_sms_status = $data['status'];

        if (isset($data['message'])) {
            $message = $data['message'];
        }

        if ($get_sms_status == null) {
            if ($sending_server->custom && $sending_server->type == 'http') {
                $cg_info = $sending_server->customSendingServer;

                $send_custom_data = [];


                $username_param = $cg_info->username_param;
                $username_value = $cg_info->username_value;
                $password_value = null;

                if ($cg_info->authorization == 'no_auth') {
                    $send_custom_data[$username_param] = $username_value;
                }

                if ($cg_info->password_status) {
                    $password_param = $cg_info->password_param;
                    $password_value = $cg_info->password_value;

                    if ($cg_info->authorization == 'no_auth') {
                        $send_custom_data[$password_param] = $password_value;
                    }
                }

                if ($cg_info->action_status) {
                    $action_param = $cg_info->action_param;
                    $action_value = $cg_info->action_value;

                    $send_custom_data[$action_param] = $action_value;
                }

                if ($cg_info->source_status) {
                    $source_param = $cg_info->source_param;
                    $source_value = $cg_info->source_value;

                    if ($data['sender_id'] != null || $data['sender_id'] != '') {
                        $send_custom_data[$source_param] = $data['sender_id'];
                    } else {
                        $send_custom_data[$source_param] = $source_value;
                    }
                }

                $destination_param                    = $cg_info->destination_param;
                $send_custom_data[$destination_param] = $data['phone'];

                $message_param                    = $cg_info->message_param;
                $send_custom_data[$message_param] = $data['message'];

                if ($cg_info->unicode_status && $data['sms_type'] == 'unicode') {
                    $unicode_param                    = $cg_info->unicode_param;
                    $unicode_value                    = $cg_info->unicode_value;
                    $send_custom_data[$unicode_param] = $unicode_value;
                }

                if ($cg_info->route_status) {
                    $route_param = $cg_info->route_param;
                    $route_value = $cg_info->route_value;

                    $send_custom_data[$route_param] = $route_value;
                }

                if ($cg_info->language_status) {
                    $language_param = $cg_info->language_param;
                    $language_value = $cg_info->language_value;

                    $send_custom_data[$language_param] = $language_value;
                }

                if ($cg_info->custom_one_status) {
                    $custom_one_param = $cg_info->custom_one_param;
                    $custom_one_value = $cg_info->custom_one_value;

                    $send_custom_data[$custom_one_param] = $custom_one_value;
                }

                if ($cg_info->custom_two_status) {
                    $custom_two_param = $cg_info->custom_two_param;
                    $custom_two_value = $cg_info->custom_two_value;

                    $send_custom_data[$custom_two_param] = $custom_two_value;
                }

                if ($cg_info->custom_three_status) {
                    $custom_three_param = $cg_info->custom_three_param;
                    $custom_three_value = $cg_info->custom_three_value;

                    $send_custom_data[$custom_three_param] = $custom_three_value;
                }

                //if json encoded then encode custom data json_encode($send_custom_data) otherwise do http_build_query
                if ($cg_info->json_encoded_post) {
                    $parameters = json_encode($send_custom_data);
                } else {
                    $parameters = http_build_query($send_custom_data);
                }

                $ch = curl_init();

                //if http method get
                if ($cg_info->http_request_method == 'get') {
                    $gateway_url = $sending_server->api_link.'?'.$parameters;

                    curl_setopt($ch, CURLOPT_URL, $gateway_url);
                    curl_setopt($ch, CURLOPT_HTTPGET, 1);
                } else {

                    //if http method post
                    $gateway_url = $sending_server->api_link;

                    curl_setopt($ch, CURLOPT_URL, $gateway_url);
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $parameters);
                }

                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

                // if ssl verify ignore set yes then add these two values in curl  curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false); curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
                if ($cg_info->ssl_certificate_verification) {
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                }
                $headers = [];
                //if content type value not none then insert content type in curl headers. $headers[] = "Content-Type: application/x-www-form-urlencoded";
                if ($cg_info->content_type != 'none') {
                    $headers[] = "Content-Type: ".$cg_info->content_type;
                }

                //if content type accept value not none then insert content type accept in curl headers. $headers[] = "Accept: application/json";
                if ($cg_info->content_type_accept != 'none') {
                    $headers[] = "Accept: ".$cg_info->content_type_accept;
                }

                //if content encoding value not none then insert content type accept in curl headers. $headers[] = "charset=utf-8";
                if ($cg_info->character_encoding != 'none') {
                    $headers[] = "charset=".$cg_info->character_encoding;
                }
                // if authorization set Bearer then add this line on curl header $header[] = "Authorization: Bearer ".$gateway_user_name;

                if ($cg_info->authorization == 'bearer_token') {
                    $headers[] = "Authorization: Bearer ".$username_value;
                }

                // if authorization set basic auth then add this line on curl header $header[] = "Authorization: Basic ".base64_encode("$gateway_user_name:$gateway_password");

                if ($cg_info->authorization == 'basic_auth') {
                    $headers[] = "Authorization: Basic ".base64_encode("$username_value:$password_value");
                }

                if (count($headers)) {
                    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                }

                $get_sms_status = curl_exec($ch);

                if (curl_errno($ch)) {
                    $get_sms_status = curl_error($ch);
                } else {
                    if (substr_count(strtolower($get_sms_status), strtolower($sending_server->success_keyword)) == 1) {
                        $get_sms_status = 'Delivered';
                    }
                }
                curl_close($ch);
            } elseif ($sending_server->type == 'smpp') {

                $sender_id = $data['sender_id'];
                $phone     = $data['phone'];
                $message   = $data['message'];

                if ($sending_server->source_addr_ton != 5) {
                    $source_ton = $sending_server->source_addr_ton;
                } else {
                    $source_ton = SMPP::TON_ALPHANUMERIC;
                }

                if ($sending_server->source_addr_ton != 5) {
                    $source_ton = $sending_server->source_addr_ton;
                } elseif (ctype_digit($sender_id) && strlen($sender_id) <= 8) {
                    $source_ton = SMPP::TON_NETWORKSPECIFIC;
                } elseif (ctype_digit($sender_id) && (strlen($sender_id) <= 15 && strlen($sender_id) >= 10)) {
                    $source_ton = SMPP::TON_INTERNATIONAL;
                } else {
                    $source_ton = SMPP::TON_ALPHANUMERIC;
                }

                if ($sending_server->dest_addr_ton != 1) {
                    $destination_ton = $sending_server->dest_addr_ton;
                } else {
                    $destination_ton = SMPP::TON_INTERNATIONAL;
                }

                try {
                    $output = (new SmsBuilder($sending_server->api_link, $sending_server->port, $sending_server->username, $sending_server->password, 10000))
                            ->setSender($data['sender_id'], $source_ton)
                            ->setRecipient($phone, $destination_ton)
                            ->sendMessage($message);

                    if ($output) {
                        $get_sms_status = 'Delivered';
                    } else {
                        $get_sms_status = __('locale.labels.failed');
                    }
                } catch (Exception $e) {
                    $get_sms_status = $e->getMessage();
                }

            } else {

                $gateway_url = $sending_server->api_link;

                switch ($gateway_name) {

                    case 'Twilio':

                        try {
                            $client       = new Client($sending_server->account_sid, $sending_server->auth_token);
                            $get_response = $client->messages->create($phone, [
                                    'from'           => $data['sender_id'],
                                    'body'           => $message,
                                    'statusCallback' => route('dlr.twilio'),
                            ]);

                            if ($get_response->status == 'queued' || $get_response->status == 'accepted') {
                                $get_sms_status = 'Delivered|'.$get_response->sid;
                            } else {
                                $get_sms_status = $get_response->status.'|'.$get_response->sid;
                            }

                        } catch (ConfigurationException | TwilioException $e) {
                            $get_sms_status = $e->getMessage();
                        }
                        break;

                    case 'TwilioCopilot':

                        try {
                            $client       = new Client($sending_server->account_sid, $sending_server->auth_token);
                            $get_response = $client->messages->create($phone, [
                                    'messagingServiceSid' => $data['sender_id'],
                                    'body'                => $message,
                            ]);

                            if ($get_response->status == 'queued' || $get_response->status == 'accepted') {
                                $get_sms_status = 'Delivered|'.$get_response->sid;
                            } else {
                                $get_sms_status = $get_response->status.'|'.$get_response->sid;
                            }

                        } catch (ConfigurationException | TwilioException $e) {
                            $get_sms_status = $e->getMessage();
                        }
                        break;

                    case 'ClickatellTouch':

                        $parameters = [
                                'api_key' => $sending_server->api_key,
                                'to'      => $phone,
                                'content' => $message,
                        ];
                        if (isset($data['sender_id'])) {
                            $parameters['from'] = $data['sender_id'];
                        }

                        $sending_url = $gateway_url.'?'.http_build_query($parameters);

                        try {

                            $ch = curl_init();

                            curl_setopt($ch, CURLOPT_URL, $sending_url);
                            curl_setopt($ch, CURLOPT_HTTPGET, 1);
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                            $response = curl_exec($ch);

                            if (curl_errno($ch)) {
                                $get_sms_status = curl_error($ch);
                            } else {
                                $get_result = json_decode($response);

                                if (isset($get_result->messages[0]->accepted) && $get_result->messages[0]->accepted) {
                                    $get_sms_status = 'Delivered|'.$get_result->messages[0]->apiMessageId;
                                } elseif (isset($get_result->messages[0]->errorDescription) && $get_result->messages[0]->errorDescription != '') {
                                    $get_sms_status = $get_result->messages[0]->errorDescription;
                                } elseif (isset($get_result->errorDescription) && $get_result->errorDescription != '') {
                                    $get_sms_status = $get_result->errorDescription;
                                } else {
                                    $get_sms_status = 'Invalid request';
                                }
                            }

                            curl_close($ch);

                        } catch (Exception $exception) {
                            $get_sms_status = $exception->getMessage();
                        }
                        break;

                    case 'ClickatellCentral':

                        $parameters = [
                                'user'     => $sending_server->username,
                                'password' => $sending_server->password,
                                'api_id'   => $sending_server->api_key,
                                'to'       => $phone,
                                'text'     => $message,
                        ];

                        if ($sms_type == 'unicode') {
                            $parameters['unicode'] = 1;
                        } else {
                            $parameters['unicode'] = 0;
                        }

                        $sending_url = $gateway_url.'?'.http_build_query($parameters);

                        try {
                            $ch = curl_init();
                            curl_setopt($ch, CURLOPT_URL, $sending_url);
                            curl_setopt($ch, CURLOPT_HTTPGET, 1);
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                            $get_sms_status = curl_exec($ch);

                            if (curl_errno($ch)) {
                                $get_sms_status = curl_error($ch);
                            } else {
                                if (substr_count($get_sms_status, 'ID:') == 1) {
                                    $get_sms_status = 'Delivered';
                                }
                            }
                            curl_close($ch);
                        } catch (Exception $exception) {
                            $get_sms_status = $exception->getMessage();
                        }
                        break;

                    case 'RouteMobile':
                        $parameters = [
                                'username'    => $sending_server->username,
                                'password'    => $sending_server->password,
                                'source'      => $data['sender_id'],
                                'destination' => $phone,
                                'message'     => $message,
                                'dlr'         => 1,
                        ];

                        if ($sms_type == 'unicode') {
                            $parameters['type'] = 2;
                        } else {
                            $parameters['type'] = 0;
                        }

                        $sending_url = $gateway_url.'?'.http_build_query($parameters);

                        try {
                            $ch = curl_init();
                            curl_setopt($ch, CURLOPT_URL, $sending_url);
                            curl_setopt($ch, CURLOPT_HTTPGET, 1);
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                            $get_sms_status = curl_exec($ch);

                            if (curl_errno($ch)) {
                                $get_sms_status = curl_error($ch);
                            } else {

                                $get_data = explode('|', $get_sms_status);

                                if (is_array($get_data) && array_key_exists('0', $get_data)) {
                                    switch ($get_data[0]) {
                                        case '1701':
                                            $get_sms_status = 'Delivered|'.$get_data['2'];
                                            break;

                                        case '1702':
                                            $get_sms_status = 'Invalid URL';
                                            break;

                                        case '1703':
                                            $get_sms_status = 'Invalid User or Password';
                                            break;

                                        case '1704':
                                            $get_sms_status = 'Invalid Type';
                                            break;

                                        case '1705':
                                            $get_sms_status = 'Invalid SMS';
                                            break;

                                        case '1706':
                                            $get_sms_status = 'Invalid receiver';
                                            break;

                                        case '1707':
                                            $get_sms_status = 'Invalid sender';
                                            break;

                                        case '1709':
                                            $get_sms_status = 'User Validation Failed';
                                            break;

                                        case '1710':
                                            $get_sms_status = 'Internal Error';
                                            break;

                                        case '1715':
                                            $get_sms_status = 'Response Timeout';
                                            break;

                                        case '1025':
                                            $get_sms_status = 'Insufficient Credit';
                                            break;

                                        default:
                                            $get_sms_status = 'Invalid request';
                                            break;

                                    }
                                } else {
                                    $get_sms_status = 'Invalid request';
                                }
                            }
                            curl_close($ch);
                        } catch (Exception $exception) {
                            $get_sms_status = $exception->getMessage();
                        }
                        break;


                    case 'TextLocal':

                        $unique_id = time();

                        $parameters = [
                                'apikey'      => $sending_server->api_key,
                                'numbers'     => $phone,
                                'sender'      => $data['sender_id'],
                                'receipt_url' => route('dlr.textlocal'),
                                'custom'      => $unique_id,
                        ];

                        if ($sms_type == 'unicode') {
                            $parameters['unicode'] = true;
                            $message               = $this->sms_unicode($message);
                        }

                        $parameters['message'] = $message;

                        try {
                            $ch = curl_init($gateway_url);
                            curl_setopt($ch, CURLOPT_POST, true);
                            curl_setopt($ch, CURLOPT_POSTFIELDS, $parameters);
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                            $response = curl_exec($ch);
                            $err      = curl_error($ch);
                            curl_close($ch);

                            if ($err) {
                                $get_sms_status = $err;
                            } else {
                                $get_data = json_decode($response, true);

                                if (isset($get_data) && is_array($get_data) && array_key_exists('status', $get_data)) {
                                    if ($get_data['status'] == 'failure') {
                                        foreach ($get_data['errors'] as $err) {
                                            $get_sms_status = $err['message'];
                                        }
                                    } elseif ($get_data['status'] == 'success') {
                                        $get_sms_status = 'Delivered|'.$unique_id;
                                    } else {
                                        $get_sms_status = $response;
                                    }
                                } else {
                                    $get_sms_status = $response;
                                }
                            }
                        } catch (Exception $exception) {
                            $get_sms_status = $exception->getMessage();
                        }
                        break;

                    case 'Plivo':

                        $client = new RestClient($sending_server->auth_id, $sending_server->auth_token);
                        try {
                            $response = $client->messages->create(
                                    $data['sender_id'],
                                    [$phone],
                                    $message,
                                    ['url' => route('dlr.plivo')],
                            );

                            $get_sms_status = 'Delivered|'.$response->getmessageUuid(0);

                        } catch (PlivoResponseException $e) {
                            $get_sms_status = $e->getMessage();
                        }

                        break;

                    case 'PlivoPowerpack':

                        $client = new RestClient($sending_server->auth_id, $sending_server->auth_token);
                        try {
                            $response = $client->messages->create(
                                    null,
                                    [$phone],
                                    $message,
                                    ['url' => route('dlr.plivo')],
                                    $data['sender_id']
                            );

                            $get_sms_status = 'Delivered|'.$response->getmessageUuid(0);

                        } catch (PlivoResponseException $e) {
                            $get_sms_status = $e->getMessage();
                        }

                        break;

                    case 'SMSGlobal':

                        $parameters = [
                                'action'   => 'sendsms',
                                'user'     => $sending_server->username,
                                'password' => $sending_server->password,
                                'from'     => $data['sender_id'],
                                'to'       => $phone,
                                'text'     => $message,
                        ];

                        if (strlen($message) > 160) {
                            $parameters['maxsplit'] = 9;
                        }

                        $sending_url = $gateway_url.'?'.http_build_query($parameters);

                        try {

                            $ch = curl_init();

                            curl_setopt($ch, CURLOPT_URL, $sending_url);
                            curl_setopt($ch, CURLOPT_HTTPGET, 1);
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                            $get_sms_status = curl_exec($ch);
                            curl_close($ch);

                            if (substr_count($get_sms_status, 'OK') == 1) {
                                $get_sms_status = explode(':', $get_sms_status);
                                if (isset($get_sms_status) && is_array($get_sms_status) && array_key_exists('3', $get_sms_status)) {
                                    $get_sms_status = 'Delivered|'.trim($get_sms_status['3']);
                                } else {
                                    $get_sms_status = 'Delivered';
                                }
                            } else {
                                $get_sms_status = str_replace('ERROR:', '', $get_sms_status);
                            }
                        } catch (Exception $exception) {
                            $get_sms_status = $exception->getMessage();
                        }
                        break;

                    case 'BulkSMS':

                        $parameters = [
                                'auto-unicode'        => true,
                                'longMessageMaxParts' => 3,
                                'to'                  => $phone,
                                'body'                => $message,
                        ];

                        if (isset($data['sender_id'])) {
                            $parameters['from'] = $data['sender_id'];
                        }

                        try {
                            $ch      = curl_init();
                            $headers = [
                                    'Content-Type:application/json',
                                    'Authorization:Basic '.base64_encode("$sending_server->username:$sending_server->password"),
                            ];
                            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                            curl_setopt($ch, CURLOPT_URL, $gateway_url);
                            curl_setopt($ch, CURLOPT_POST, 1);
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($parameters));
                            $response = curl_exec($ch);
                            curl_close($ch);

                            $get_data = json_decode($response, true);

                            if (isset($get_data) && is_array($get_data)) {
                                if (array_key_exists('id', $get_data)) {
                                    $get_sms_status = 'Delivered|'.$get_data['id'];
                                } elseif (array_key_exists('detail', $get_data)) {
                                    $get_sms_status = $get_data['detail'];
                                }
                            } else {
                                $get_sms_status = $response;
                            }
                        } catch (Exception $exception) {
                            $get_sms_status = $exception->getMessage();
                        }
                        break;

                    case 'Vonage':

                        $client = new \Vonage\Client(new Basic($sending_server->api_key, $sending_server->api_secret));
                        $text   = new SMS($phone, $data['sender_id'], $message);

                        try {
                            $response = $client->sms()->send($text);
                            $output   = $response->current();

                            if ($output->getStatus() == 0) {
                                $get_sms_status = 'Delivered|'.$output->getMessageId();
                            } else {
                                $get_sms_status = $output->getStatus();
                            }

                        } catch (ClientExceptionInterface | \Vonage\Client\Exception\Exception $e) {
                            $get_sms_status = $e->getMessage();
                        }
                        break;

                    case 'Infobip':
                        $destination = [
                                'messageId' => time(),
                                'to'        => $phone,
                        ];

                        $parameters = [
                                'messages' => [
                                        "from"              => $data['sender_id'],
                                        "destinations"      => [$destination],
                                        'text'              => $message,
                                        'notifyUrl'         => route('dlr.infobip'),
                                        'notifyContentType' => 'application/json',
                                ],
                        ];

                        try {

                            $ch     = curl_init();
                            $header = [
                                    "Authorization: App $sending_server->api_key",
                                    "Content-Type: application/json",
                                    "Accept: application/json",
                            ];

                            // setting options
                            curl_setopt($ch, CURLOPT_URL, $gateway_url);
                            curl_setopt($ch, CURLOPT_HTTPHEADER, $header);
                            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
                            curl_setopt($ch, CURLOPT_MAXREDIRS, 2);
                            curl_setopt($ch, CURLOPT_POST, 1);
                            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($parameters));

                            // response of the POST request
                            $response = curl_exec($ch);
                            $get_data = json_decode($response, true);
                            curl_close($ch);

                            if (is_array($get_data)) {
                                if (array_key_exists('messages', $get_data)) {
                                    foreach ($get_data['messages'] as $msg) {
                                        $get_sms_status = 'Delivered|'.$msg['messageId'];
                                    }
                                } elseif (array_key_exists('requestError', $get_data)) {
                                    foreach ($get_data['requestError'] as $msg) {
                                        $get_sms_status = $msg['messageId'];
                                    }
                                } else {
                                    $get_sms_status = 'Unknown error';
                                }
                            } else {
                                $get_sms_status = 'Unknown error';
                            }

                        } catch (Exception $exception) {
                            $get_sms_status = $exception->getMessage();
                        }
                        break;

//                case 'SmsGatewayMe':
//                    $config = Configuration::getDefaultConfiguration();
//                    $config->setApiKey('Authorization', $sending_server->api_token);
//
//                    $apiClient     = new ApiClient($config);
//                    $messageClient = new MessageApi($apiClient);
//
//                    $parameters = [
//                            'phoneNumber' => $phone,
//                            'message'     => $message,
//                            'deviceId'    => $sending_server->device_id,
//                    ];
//
//                    if (isset($data['sender_id'])) {
//                        $parameters['from'] = $data['sender_id'];
//                    }
//
//                    try {
//                        $response = $messageClient->sendMessages($parameters);
//
//                        if (is_array($response)) {
//                            if (array_key_exists('0', $response)) {
//                                $get_sms_status = 'Delivered|'.$response[0]->getId();
//                            } else {
//                                $get_sms_status = 'Invalid request';
//                            }
//
//                        } else {
//                            $get_sms_status = 'Unknown Error';
//                        }
//
//                        $get_sms_status = trim($get_sms_status);
//
//                    } catch (ApiException $e) {
//                        $get_sms_status = $e->getMessage();
//                    }
//
//                    break;

                    case '1s2u':

                        if ($sms_type == 'unicode') {
                            $mt      = 1;
                            $message = bin2hex(mb_convert_encoding($message, "UTF-16", "UTF-8"));
                        } else {
                            $mt = 0;
                        }

                        $parameters = [
                                "username" => $sending_server->username,
                                "password" => $sending_server->password,
                                "mno"      => $phone,
                                "msg"      => $message,
                                "sid"      => $data['sender_id'],
                                "mt"       => $mt,
                                "fl"       => 0,
                        ];

                        $sending_url = $gateway_url.'?'.http_build_query($parameters);

                        try {

                            $ch = curl_init();
                            curl_setopt($ch, CURLOPT_URL, $sending_url);
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                            curl_setopt($ch, CURLOPT_HTTPGET, 1);

                            $get_sms_status = curl_exec($ch);

                            curl_close($ch);

                            if (strpos($get_sms_status, 'OK') !== false) {
                                $get_sms_status = 'Delivered|'.trim(str_replace('OK: ', '', $get_sms_status));
                            } else {
                                switch ($get_sms_status) {
                                    case '0005':
                                        $get_sms_status = 'Invalid Sender';
                                        break;
                                    case '0010':
                                        $get_sms_status = 'Username not provided';
                                        break;
                                    case '0011':
                                        $get_sms_status = 'Password not provided';
                                        break;
                                    case '00':
                                        $get_sms_status = 'Invalid username/password';
                                        break;
                                    case '0020':
                                        $get_sms_status = 'Insufficient Credits';
                                        break;
                                    case '0030':
                                        $get_sms_status = 'Invalid Sender ID';
                                        break;
                                    case '0040':
                                        $get_sms_status = 'Mobile number not provided';
                                        break;
                                    case '0041':
                                        $get_sms_status = 'Invalid mobile number';
                                        break;
                                    case '0066':
                                    case '0042':
                                        $get_sms_status = 'Network not supported';
                                        break;
                                    case '0050':
                                        $get_sms_status = 'Invalid message';
                                        break;
                                    case '0060':
                                        $get_sms_status = 'Invalid quantity specified';
                                        break;
                                    case '0000':
                                        $get_sms_status = 'Message not sent';
                                        break;

                                    default:
                                        $get_sms_status = 'Unknown Error';
                                        break;
                                }

                            }
                        } catch (Exception $ex) {
                            $get_sms_status = $ex->getMessage();
                        }
                        break;

                    case 'MessageBird':
                        $parameters = [
                                'recipients' => $phone,
                                'originator' => $data['sender_id'],
                                'body'       => $message,
                                'datacoding' => 'auto',
                        ];

                        $ch = curl_init();

                        curl_setopt($ch, CURLOPT_URL, $gateway_url);
                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($parameters));
                        curl_setopt($ch, CURLOPT_POST, 1);

                        $headers   = [];
                        $headers[] = "Authorization: AccessKey $sending_server->api_key";
                        $headers[] = "Content-Type: application/x-www-form-urlencoded";
                        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

                        $result = curl_exec($ch);
                        if (curl_errno($ch)) {
                            $get_sms_status = curl_error($ch);
                        }
                        curl_close($ch);

                        $response = json_decode($result, true);

                        if (is_array($response) && array_key_exists('id', $response)) {
                            $get_sms_status = 'Delivered|'.$response['id'];
                        } elseif (is_array($response) && array_key_exists('errors', $response)) {
                            $get_sms_status = $response['errors'][0]['description'];
                        } else {
                            $get_sms_status = 'Unknown Error';
                        }
                        break;

                    case 'AmazonSNS':
                        $credentials = [
                                'credentials' => [
                                        'key'    => $sending_server->access_key,
                                        'secret' => $sending_server->secret_access,
                                ],
                                'region'      => $sending_server->region, // < your aws from SNS Topic region
                                'version'     => 'latest',
                        ];

                        $sns = new SnsClient($credentials);

                        $parameters = [
                                'MessageAttributes' => [
                                        'AWS.SNS.SMS.SenderID' => [
                                                'DataType'    => 'String',
                                                'StringValue' => $data['sender_id'],
                                        ],
                                ],
                                "SMSType"           => $sending_server->route,
                                "PhoneNumber"       => '+'.$phone,
                                "Message"           => $message,
                        ];

                        try {
                            $result = $sns->publish($parameters)->toArray();
                            if (is_array($result) && array_key_exists('MessageId', $result)) {
                                $get_sms_status = 'Delivered|'.$result['MessageId'];
                            } else {
                                $get_sms_status = 'Unknown error';
                            }
                        } catch (SnsException $exception) {
                            $get_sms_status = $exception->getAwsErrorMessage();
                        }

                        break;

                    case 'Tyntec':
                        $parameters = [
                                'from'    => $data['sender_id'],
                                'to'      => $phone,
                                'message' => $message,
                        ];

                        try {
                            $ch = curl_init();

                            curl_setopt($ch, CURLOPT_URL, $gateway_url);
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($parameters));
                            curl_setopt($ch, CURLOPT_POST, 1);
                            curl_setopt($ch, CURLOPT_USERPWD, "$sending_server->username".":"."$sending_server->password");

                            $headers   = [];
                            $headers[] = "Content-Type: application/json";
                            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

                            $result = curl_exec($ch);
                            curl_close($ch);
                            $result = json_decode($result, true);

                            if (is_array($result) && array_key_exists('requestId', $result)) {
                                $get_sms_status = 'Delivered';
                            } elseif (is_array($result) && array_key_exists('status', $result)) {
                                $get_sms_status = $result['message'];
                            } else {
                                $get_sms_status = 'Invalid request';
                            }

                        } catch (Exception $e) {
                            $get_sms_status = $e->getMessage();
                        }
                        break;


                    case 'KarixIO':
                        $parameters = [
                                'channel'     => 'sms',
                                'source'      => $data['sender_id'],
                                'destination' => [$phone],
                                'content'     => [
                                        'text' => $message,
                                ],
                        ];

                        try {

                            $headers = [
                                    'Content-Type:application/json',
                            ];

                            $ch = curl_init();
                            curl_setopt($ch, CURLOPT_URL, $gateway_url);
                            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                            curl_setopt($ch, CURLOPT_USERPWD, "$sending_server->auth_id".":"."$sending_server->auth_token");
                            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                            curl_setopt($ch, CURLOPT_POST, 1);
                            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($parameters));
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                            $response = curl_exec($ch);
                            curl_close($ch);

                            $get_response = json_decode($response, true);

                            if (isset($get_response) && is_array($get_response)) {
                                if (array_key_exists('objects', $get_response)) {
                                    if ($get_response['objects']['0']['status'] == 'queued') {
                                        $get_sms_status = 'Delivered|'.$get_response['objects']['0']['account_uid'];
                                    } else {
                                        $get_sms_status = $get_response['objects']['0']['status'];
                                    }
                                } elseif (array_key_exists('error', $get_response)) {
                                    $get_sms_status = $get_response['error']['message'];
                                } else {
                                    $get_sms_status = 'Unknown error';
                                }
                            } else {
                                $get_sms_status = 'Unknown error';
                            }

                        } catch (Exception $e) {
                            $get_sms_status = $e->getMessage();
                        }
                        break;

                    case 'SignalWire':

                        $parameters = [
                                'From' => '+'.$data['sender_id'],
                                'Body' => $message,
                                'To'   => '+'.$phone,
                        ];

                        $sending_url = $gateway_url."/api/laml/2010-04-01/Accounts/$sending_server->api_token/Messages.json";

                        $ch = curl_init();

                        curl_setopt($ch, CURLOPT_URL, $sending_url);
                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                        curl_setopt($ch, CURLOPT_POST, 1);
                        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($parameters));
                        curl_setopt($ch, CURLOPT_USERPWD, "$sending_server->api_token".":"."$sending_server->project_id");

                        $get_response = curl_exec($ch);
                        if (curl_errno($ch)) {
                            $get_sms_status = curl_error($ch);
                        }
                        curl_close($ch);

                        $result = json_decode($get_response, true);

                        if (isset($result) && is_array($result) && array_key_exists('status', $result) && array_key_exists('error_code', $result)) {
                            if ($result['status'] == 'queued' && $result['error_code'] === null) {
                                $get_sms_status = 'Delivered|'.$result['sid'];
                            } else {
                                $get_sms_status = $result['error_message'];
                            }
                        } elseif (isset($result) && is_array($result) && array_key_exists('status', $result) && array_key_exists('message', $result)) {
                            $get_sms_status = $result['message'];
                        } else {
                            $get_sms_status = $get_response;
                        }

                        if ($get_sms_status === null) {
                            $get_sms_status = 'Check your settings';
                        }
                        break;

                    case 'Telnyx':

                        if (is_numeric($data['sender_id'])) {
                            $sender_id = '+'.$data['sender_id'];
                        } else {
                            $sender_id = $$data['sender_id'];
                        }

                        $parameters = [
                                "to"   => '+'.$phone,
                                "text" => $message,
                        ];

                        if (strlen($sender_id) > 12) {
                            $parameters['messaging_profile_id'] = $sender_id;
                        } else {
                            $parameters['from'] = $sender_id;
                        }

                        try {

                            $headers = [
                                    'Content-Type:application/json',
                                    'Authorization: Bearer '.$sending_server->api_key,
                            ];

                            $ch = curl_init();
                            curl_setopt($ch, CURLOPT_URL, $gateway_url);
                            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                            curl_setopt($ch, CURLOPT_POST, 1);
                            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($parameters));
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                            $response = curl_exec($ch);
                            curl_close($ch);

                            $get_response = json_decode($response, true);

                            if (isset($get_response) && is_array($get_response)) {
                                if (array_key_exists('data', $get_response) && array_key_exists('to', $get_response['data']) && $get_response['data']['to'][0]['status'] == 'queued') {
                                    $get_sms_status = 'Delivered';
                                } elseif (array_key_exists('errors', $get_response)) {
                                    $get_sms_status = $get_response['errors'][0]['detail'];
                                } else {
                                    $get_sms_status = (string) $response;
                                }
                            } else {
                                $get_sms_status = 'Unknown error';
                            }

                        } catch (Exception $e) {
                            $get_sms_status = $e->getMessage();
                        }
                        break;

                    case 'Bandwidth':

                        $parameters = [
                                'from'          => '+'.$data['sender_id'],
                                'to'            => ['+'.$phone],
                                'text'          => $message,
                                'applicationId' => $sending_server->application_id,
                        ];

                        try {

                            $ch = curl_init();

                            curl_setopt($ch, CURLOPT_URL, $gateway_url);
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($parameters));
                            curl_setopt($ch, CURLOPT_POST, 1);
                            curl_setopt($ch, CURLOPT_USERPWD, $sending_server->api_token.':'.$sending_server->api_secret);

                            $headers   = [];
                            $headers[] = 'Content-Type: application/json';
                            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

                            $result = curl_exec($ch);

                            if (curl_errno($ch)) {
                                $get_sms_status = curl_error($ch);
                            } else {

                                $result = json_decode($result, true);

                                if (isset($result) && is_array($result)) {
                                    if (array_key_exists('id', $result)) {
                                        $get_sms_status = 'Delivered|'.$result['id'];
                                    } elseif (array_key_exists('error', $result)) {
                                        $get_sms_status = $result['error'];
                                    } elseif (array_key_exists('fieldErrors', $result)) {
                                        $get_sms_status = $result['fieldErrors'][0]['fieldName'].' '.$result['fieldErrors'][0]['description'];
                                    } else {
                                        $get_sms_status = implode(" ", $result);
                                    }
                                } else {
                                    $get_sms_status = $result;
                                }
                            }
                            curl_close($ch);
                        } catch (Exception $e) {
                            $get_sms_status = $e->getMessage();
                        }
                        break;

                    default:
                        $get_sms_status = __('locale.sending_servers.sending_server_not_found');
                        break;
                }
            }
        }

        $reportsData = [
                'user_id'           => $data['user_id'],
                'to'                => $phone,
                'message'           => $message,
                'sms_type'          => $data['sms_type'],
                'status'            => $get_sms_status,
                'cost'              => $data['cost'],
                'sending_server_id' => $sending_server->id,
        ];

        if (isset($data['sender_id'])) {
            $reportsData['from'] = $data['sender_id'];
        }

        if (isset($data['campaign_id'])) {
            $reportsData['campaign_id'] = $data['campaign_id'];
        }

        if (isset($data['api_key'])) {
            $reportsData['api_key'] = $data['api_key'];
            $reportsData['send_by'] = 'api';
        } else {
            $reportsData['send_by'] = 'from';
        }

        $status = Reports::create($reportsData);

        if ($status) {
            return $status;
        }

        return __('locale.exceptions.something_went_wrong');

    }


    /**
     * send voice message
     *
     * @param $data
     *
     * @return array|Application|Translator|string|null
     */
    public function sendVoiceSMS($data)
    {
        $phone          = $data['phone'];
        $sending_server = $data['sending_server'];
        $gateway_name   = $data['sending_server']->settings;
        $message        = null;
        $sms_type       = 'voice';
        $get_sms_status = $data['status'];
        $language       = $data['language'];
        $gender         = $data['gender'];

        if (isset($data['message'])) {
            $message = $data['message'];
        }
        if ($get_sms_status == null) {
            $gateway_url = $sending_server->api_link;
            switch ($gateway_name) {

                case 'Twilio':

                    try {
                        $client = new Client($sending_server->account_sid, $sending_server->auth_token);

                        $get_response = $client->calls->create($phone, $data['sender_id'], [
                                "url" => Tool::createVoiceFile($message, 'Twilio'),
                        ]);

                        if ($get_response->status == 'queued') {
                            $get_sms_status = 'Delivered';
                        } else {
                            $get_sms_status = $get_response->status.'|'.$get_response->sid;
                        }

                    } catch (ConfigurationException | TwilioException $e) {
                        $get_sms_status = $e->getMessage();
                    }
                    break;

                case 'Plivo':

                    $client = new RestClient($sending_server->auth_id, $sending_server->auth_token);
                    try {
                        $response = $client->messages->create(
                                $data['sender_id'],
                                [$phone],
                                Tool::createVoiceFile($message, 'Plivo'),
                        );

                        $get_sms_status = 'Delivered|'.$response->getmessageUuid(0);

                    } catch (PlivoResponseException $e) {
                        $get_sms_status = $e->getMessage();
                    }

                    break;


                case 'Infobip':

                    $parameters = [
                            'text'     => $message,
                            'language' => $data['language'],
                            'voice'    => [
                                    'gender' => $data['gender'],
                            ],
                            'from'     => $data['sender_id'],
                            'to'       => $phone,
                    ];
                    try {
                        $curl = curl_init();

                        $header = [
                                "Authorization: App $sending_server->api_key",
                                "Content-Type: application/json",
                                "Accept: application/json",
                        ];

                        curl_setopt_array($curl, [
                                CURLOPT_URL            => 'https://api.infobip.com/tts/3/single',
                                CURLOPT_RETURNTRANSFER => true,
                                CURLOPT_ENCODING       => '',
                                CURLOPT_MAXREDIRS      => 10,
                                CURLOPT_TIMEOUT        => 0,
                                CURLOPT_FOLLOWLOCATION => true,
                                CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
                                CURLOPT_CUSTOMREQUEST  => 'POST',
                                CURLOPT_POSTFIELDS     => json_encode($parameters),
                                CURLOPT_HTTPHEADER     => $header,
                        ]);


                        // response of the POST request
                        $response = curl_exec($curl);
                        $get_data = json_decode($response, true);
                        curl_close($curl);

                        if (is_array($get_data)) {
                            if (array_key_exists('messages', $get_data)) {
                                foreach ($get_data['messages'] as $msg) {
                                    if ($msg['status']['name'] == 'MESSAGE_ACCEPTED' || $msg['status']['name'] == 'PENDING_ENROUTE' || $msg['status']['name'] == 'PENDING_ACCEPTED') {
                                        $get_sms_status = 'Delivered|'.$msg['messageId'];
                                    } else {
                                        $get_sms_status = $msg['status']['description'];
                                    }
                                }
                            } elseif (array_key_exists('requestError', $get_data)) {
                                foreach ($get_data['requestError'] as $msg) {
                                    $get_sms_status = $msg['text'];
                                }
                            } else {
                                $get_sms_status = 'Unknown error';
                            }
                        } else {
                            $get_sms_status = 'Unknown error';
                        }

                    } catch (Exception $exception) {
                        $get_sms_status = $exception->getMessage();
                    }
                    break;

                case 'MessageBird':
                    $parameters = [
                            'destination' => $phone,
                            'source'      => $data['sender_id'],
                            'callFlow'    => [
                                    'title' => config('app.name').'_'.now().'_flow',
                                    'steps' => [
                                            [
                                                    'action'  => 'say',
                                                    'options' => [
                                                            'payload'  => $message,
                                                            'language' => $data['language'],
                                                            'voice'    => $data['gender'],
                                                    ],
                                            ],
                                    ],
                            ],
                    ];

                    $ch = curl_init();

                    curl_setopt($ch, CURLOPT_URL, 'https://voice.messagebird.com/calls');
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($parameters));
                    curl_setopt($ch, CURLOPT_POST, 1);

                    $headers   = [];
                    $headers[] = "Authorization: AccessKey $sending_server->api_key";
                    $headers[] = "Content-Type: application/x-www-form-urlencoded";
                    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

                    $result = curl_exec($ch);
                    if (curl_errno($ch)) {
                        $get_sms_status = curl_error($ch);
                    }
                    curl_close($ch);

                    $response = json_decode($result, true);

                    if (is_array($response) && array_key_exists('data', $response)) {
                        $get_sms_status = 'Delivered|'.$response['data'][0]['id'];
                    } elseif (is_array($response) && array_key_exists('errors', $response)) {
                        $get_sms_status = $response['errors'][0]['message'];
                    } else {
                        $get_sms_status = 'Unknown Error';
                    }
                    break;

                case 'SignalWire':

                    $parameters = [
                            'From' => '+'.$data['sender_id'],
                            'Url'  => Tool::createVoiceFile($message, 'Twilio'),
                            'To'   => '+'.$phone,
                    ];

                    $sending_url = $gateway_url."/api/laml/2010-04-01/Accounts/$sending_server->api_token/Calls.json";

                    $ch = curl_init();

                    curl_setopt($ch, CURLOPT_URL, $sending_url);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                    curl_setopt($ch, CURLOPT_POST, 1);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($parameters));
                    curl_setopt($ch, CURLOPT_USERPWD, "$sending_server->api_token".":"."$sending_server->project_id");

                    $get_response = curl_exec($ch);
                    if (curl_errno($ch)) {
                        $get_sms_status = curl_error($ch);
                    }
                    curl_close($ch);

                    $result = json_decode($get_response, true);

                    if (isset($result) && is_array($result) && array_key_exists('status', $result) && array_key_exists('error_code', $result)) {
                        if ($result['status'] == 'queued' && $result['error_code'] === null) {
                            $get_sms_status = 'Delivered|'.$result['sid'];
                        } else {
                            $get_sms_status = $result['error_message'];
                        }
                    } elseif (isset($result) && is_array($result) && array_key_exists('status', $result) && array_key_exists('message', $result)) {
                        $get_sms_status = $result['message'];
                    } else {
                        $get_sms_status = $get_response;
                    }

                    if ($get_sms_status === null) {
                        $get_sms_status = 'Check your settings';
                    }
                    break;

                default:
                    $get_sms_status = __('locale.sending_servers.sending_server_not_found');
                    break;
            }
        }


        $reportsData = [
                'user_id'           => $data['user_id'],
                'to'                => $phone,
                'message'           => $message,
                'sms_type'          => 'voice',
                'status'            => $get_sms_status,
                'cost'              => $data['cost'],
                'sending_server_id' => $sending_server->id,
        ];

        if (isset($data['sender_id'])) {
            $reportsData['from'] = $data['sender_id'];
        }

        if (isset($data['campaign_id'])) {
            $reportsData['campaign_id'] = $data['campaign_id'];
        }

        if (isset($data['api_key'])) {
            $reportsData['api_key'] = $data['api_key'];
            $reportsData['send_by'] = 'api';
        } else {
            $reportsData['send_by'] = 'from';
        }

        $status = Reports::create($reportsData);

        if ($status) {
            return $status;
        }

        return __('locale.exceptions.something_went_wrong');

    }

    /**
     * send mms message
     *
     * @param $data
     *
     * @return array|Application|Translator|string|null
     */
    public function sendMMS($data)
    {
        $phone          = $data['phone'];
        $sending_server = $data['sending_server'];
        $gateway_name   = $data['sending_server']->settings;
        $message        = null;
        $sms_type       = 'mms';
        $get_sms_status = $data['status'];
        $media_url      = $data['media_url'];

        if (isset($data['message'])) {
            $message = $data['message'];
        }
        if ($get_sms_status == null) {
            $gateway_url = $sending_server->api_link;
            switch ($gateway_name) {

                case 'Twilio':

                    try {
                        $client = new Client($sending_server->account_sid, $sending_server->auth_token);

                        $get_response = $client->messages->create($phone, [
                                'from'     => $data['sender_id'],
                                'body'     => $message,
                                'mediaUrl' => $media_url,
                        ]);

                        if ($get_response->status == 'queued' || $get_response->status == 'accepted') {
                            $get_sms_status = 'Delivered|'.$get_response->sid;
                        } else {
                            $get_sms_status = $get_response->status.'|'.$get_response->sid;
                        }

                    } catch (ConfigurationException | TwilioException $e) {
                        $get_sms_status = $e->getMessage();
                    }
                    break;
                case 'TextLocal':

                    $parameters = [
                            'apikey'  => $sending_server->api_key,
                            'numbers' => $phone,
                            'url'     => $media_url,
                            'message' => $message,
                    ];

                    if (isset($data['sender_id'])) {
                        $parameters['sender'] = $data['sender_id'];
                    }

                    try {
                        $ch = curl_init("https://api.txtlocal.com/send_mms/");
                        curl_setopt($ch, CURLOPT_POST, true);
                        curl_setopt($ch, CURLOPT_POSTFIELDS, $parameters);
                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                        $response = curl_exec($ch);
                        $err      = curl_error($ch);
                        curl_close($ch);

                        if ($err) {
                            $get_sms_status = $err;
                        } else {
                            $get_data = json_decode($response, true);

                            if (isset($get_data) && is_array($get_data) && array_key_exists('status', $get_data)) {
                                if ($get_data['status'] == 'failure') {
                                    foreach ($get_data['errors'] as $err) {
                                        $get_sms_status = $err['message'];
                                    }
                                } elseif ($get_data['status'] == 'success') {
                                    $get_sms_status = 'Delivered';
                                } else {
                                    $get_sms_status = $response;
                                }
                            } else {
                                $get_sms_status = $response;
                            }
                        }
                    } catch (Exception $exception) {
                        $get_sms_status = $exception->getMessage();
                    }
                    break;

                case 'Plivo':

                    $client = new RestClient($sending_server->auth_id, $sending_server->auth_token);
                    try {
                        $response = $client->messages->create(
                                $data['sender_id'],
                                [$phone],
                                $message,
                                ['media_urls' => $media_url]
                        );

                        $get_sms_status = 'Delivered|'.$response->getmessageUuid(0);

                    } catch (PlivoResponseException $e) {
                        $get_sms_status = $e->getMessage();
                    }

                    break;


                case 'PlivoPowerpack':

                    $client = new RestClient($sending_server->auth_id, $sending_server->auth_token);
                    try {
                        $response = $client->messages->create(
                                null,
                                [$phone],
                                $message,
                                ['media_urls' => $media_url],
                                $data['sender_id']
                        );

                        $get_sms_status = 'Delivered|'.$response->getmessageUuid(0);

                    } catch (PlivoResponseException $e) {
                        $get_sms_status = $e->getMessage();
                    }

                    break;


                case 'SMSGlobal':

                    $parameters = [
                            'user'        => $sending_server->username,
                            'password'    => $sending_server->password,
                            'from'        => $data['sender_id'],
                            'number'      => $phone,
                            'message'     => $message,
                            'attachmentx' => $media_url,
                            'typex'       => mime_content_type($media_url),
                            'namex'       => basename($media_url),
                    ];

                    $sending_url = 'https://api.smsglobal.com/mms/sendmms.php?'.http_build_query($parameters);

                    try {

                        $ch = curl_init();

                        curl_setopt($ch, CURLOPT_URL, $sending_url);
                        curl_setopt($ch, CURLOPT_HTTPGET, 1);
                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                        $get_sms_status = curl_exec($ch);
                        curl_close($ch);

                        if (substr_count($get_sms_status, 'SUCCESS')) {
                            $get_sms_status = 'Delivered';
                        } else {
                            $get_sms_status = str_replace('ERROR:', '', $get_sms_status);
                        }

                    } catch (Exception $exception) {
                        $get_sms_status = $exception->getMessage();
                    }
                    break;


                case 'MessageBird':
                    $parameters = [
                            'recipients' => $data['phone'],
                            'originator' => $data['sender_id'],
                            'body'       => $message,
                            'mediaUrls'  => [$media_url],
                    ];

                    $ch = curl_init();

                    curl_setopt($ch, CURLOPT_URL, 'https://rest.messagebird.com/mms');
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($parameters));
                    curl_setopt($ch, CURLOPT_POST, 1);

                    $headers   = [];
                    $headers[] = "Authorization: AccessKey $sending_server->api_key";
                    $headers[] = "Content-Type: application/x-www-form-urlencoded";
                    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

                    $result = curl_exec($ch);
                    if (curl_errno($ch)) {
                        $get_sms_status = curl_error($ch);
                    }
                    curl_close($ch);

                    $response = json_decode($result, true);

                    if (is_array($response) && array_key_exists('id', $response)) {
                        $get_sms_status = 'Delivered|'.$response['id'];
                    } elseif (is_array($response) && array_key_exists('errors', $response)) {
                        $get_sms_status = $response['errors'][0]['description'];
                    } else {
                        $get_sms_status = 'Unknown Error';
                    }
                    break;


                case 'SignalWire':

                    $parameters = [
                            'From'     => '+'.$data['sender_id'],
                            'Body'     => $message,
                            'MediaUrl' => $media_url,
                            'To'       => '+'.$phone,
                    ];

                    $sending_url = $gateway_url."/api/laml/2010-04-01/Accounts/$sending_server->api_token/Messages.json";

                    $ch = curl_init();

                    curl_setopt($ch, CURLOPT_URL, $sending_url);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                    curl_setopt($ch, CURLOPT_POST, 1);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($parameters));
                    curl_setopt($ch, CURLOPT_USERPWD, "$sending_server->api_token".":"."$sending_server->project_id");

                    $get_response = curl_exec($ch);
                    if (curl_errno($ch)) {
                        $get_sms_status = curl_error($ch);
                    }
                    curl_close($ch);

                    $result = json_decode($get_response, true);

                    if (isset($result) && is_array($result) && array_key_exists('status', $result) && array_key_exists('error_code', $result)) {
                        if ($result['status'] == 'queued' && $result['error_code'] === null) {
                            $get_sms_status = 'Delivered|'.$result['sid'];
                        } else {
                            $get_sms_status = $result['error_message'];
                        }
                    } elseif (isset($result) && is_array($result) && array_key_exists('status', $result) && array_key_exists('message', $result)) {
                        $get_sms_status = $result['message'];
                    } else {
                        $get_sms_status = $get_response;
                    }

                    if ($get_sms_status === null) {
                        $get_sms_status = 'Check your settings';
                    }
                    break;

                case 'Telnyx':
                    if (is_numeric($data['sender_id'])) {
                        $sender_id = '+'.$data['sender_id'];
                    } else {
                        $sender_id = $$data['sender_id'];
                    }

                    $parameters = [
                            "to"         => '+'.$phone,
                            "text"       => $message,
                            "subject"    => 'Picture',
                            "media_urls" => [$media_url],
                    ];

                    if (strlen($sender_id) > 12) {
                        $parameters['messaging_profile_id'] = $sender_id;
                    } else {
                        $parameters['from'] = $sender_id;
                    }

                    try {

                        $headers = [
                                'Content-Type:application/json',
                                'Authorization: Bearer '.$sending_server->api_key,
                        ];

                        $ch = curl_init();
                        curl_setopt($ch, CURLOPT_URL, $gateway_url);
                        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                        curl_setopt($ch, CURLOPT_POST, 1);
                        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($parameters));
                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                        $response = curl_exec($ch);
                        curl_close($ch);

                        $get_response = json_decode($response, true);

                        if (isset($get_response) && is_array($get_response)) {
                            if (array_key_exists('data', $get_response) && array_key_exists('to', $get_response['data']) && $get_response['data']['to'][0]['status'] == 'queued') {
                                $get_sms_status = 'Delivered';
                            } elseif (array_key_exists('errors', $get_response)) {
                                $get_sms_status = $get_response['errors'][0]['detail'];
                            } else {
                                $get_sms_status = (string) $response;
                            }
                        } else {
                            $get_sms_status = 'Unknown error';
                        }

                    } catch (Exception $e) {
                        $get_sms_status = $e->getMessage();
                    }
                    break;

                case 'Bandwidth':


                    $sender_id = str_replace(['+', '(', ')', '-', ' '], '', $this->sender_id);

                    $parameters = [
                            'from'          => '+'.$sender_id,
                            'to'            => ['+'.$this->cl_phone],
                            'text'          => $this->message,
                            'applicationId' => $sending_server->application_id,
                            'media'         => [
                                    $this->media_url,
                            ],
                    ];

                    try {

                        $ch = curl_init();

                        curl_setopt($ch, CURLOPT_URL, $gateway_url);
                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($parameters));
                        curl_setopt($ch, CURLOPT_POST, 1);
                        curl_setopt($ch, CURLOPT_USERPWD, $sending_server->api_secret.':'.$sending_server->api_token);

                        $headers   = [];
                        $headers[] = 'Content-Type: application/json';
                        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

                        $result = curl_exec($ch);

                        if (curl_errno($ch)) {
                            $get_sms_status = curl_error($ch);
                        } else {

                            $result = json_decode($result, true);

                            if (isset($result) && is_array($result)) {
                                if (array_key_exists('id', $result)) {
                                    $get_sms_status = 'Success|'.$result['id'];
                                } elseif (array_key_exists('error', $result)) {
                                    $get_sms_status = $result['error'];
                                } elseif (array_key_exists('fieldErrors', $result)) {
                                    $get_sms_status = $result['fieldErrors'][0]['fieldName'].' '.$result['fieldErrors'][0]['description'];
                                } else {
                                    $get_sms_status = implode(" ", $result);
                                }
                            } else {
                                $get_sms_status = $result;
                            }
                        }
                        curl_close($ch);
                    } catch (Exception $e) {
                        $get_sms_status = $e->getMessage();
                    }
                    break;

                default:
                    $get_sms_status = __('locale.sending_servers.sending_server_not_found');
                    break;
            }
        }


        $reportsData = [
                'user_id'           => $data['user_id'],
                'to'                => $phone,
                'message'           => $message,
                'sms_type'          => 'mms',
                'status'            => $get_sms_status,
                'cost'              => $data['cost'],
                'sending_server_id' => $sending_server->id,
                'media_url'         => $media_url,
        ];

        if (isset($data['sender_id'])) {
            $reportsData['from'] = $data['sender_id'];
        }

        if (isset($data['campaign_id'])) {
            $reportsData['campaign_id'] = $data['campaign_id'];
        }

        if (isset($data['api_key'])) {
            $reportsData['api_key'] = $data['api_key'];
            $reportsData['send_by'] = 'api';
        } else {
            $reportsData['send_by'] = 'from';
        }

        $status = Reports::create($reportsData);

        if ($status) {
            return $status;
        }

        return __('locale.exceptions.something_went_wrong');

    }


    /**
     * send whatsapp message
     *
     * @param $data
     *
     * @return array|Application|Translator|string|null
     */
    public function sendWhatsApp($data)
    {
        $phone          = $data['phone'];
        $sending_server = $data['sending_server'];
        $gateway_name   = $data['sending_server']->settings;
        $message        = null;
        $sms_type       = 'whatsapp';
        $get_sms_status = $data['status'];
        $message        = $data['message'];

        if ($get_sms_status == null) {
            $gateway_url = $sending_server->api_link;
            switch ($gateway_name) {

                case 'Twilio':

                    try {
                        $client = new Client($sending_server->account_sid, $sending_server->auth_token);

                        $get_response = $client->messages->create(
                                'whatsapp:'.$phone, [
                                        'from' => 'whatsapp:'.$data['sender_id'],
                                        'body' => $message,
                                ]
                        );

                        if ($get_response->status == 'queued' || $get_response->status == 'accepted') {
                            $get_sms_status = 'Delivered|'.$get_response->sid;
                        } else {
                            $get_sms_status = $get_response->status.'|'.$get_response->sid;
                        }

                    } catch (ConfigurationException | TwilioException $e) {
                        $get_sms_status = $e->getMessage();
                    }
                    break;

                case 'MessageBird':
                    $parameters = [
                            'to'      => $data['phone'],
                            'from'    => $data['sender_id'],
                            'type'    => 'text',
                            'content' => [
                                    'text' => $message,
                            ],
                    ];

                    $ch = curl_init();

                    curl_setopt($ch, CURLOPT_URL, 'https://conversations.messagebird.com/v1/send');
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($parameters));
                    curl_setopt($ch, CURLOPT_POST, 1);

                    $headers   = [];
                    $headers[] = "Authorization: AccessKey $sending_server->api_key";
                    $headers[] = "Content-Type: application/x-www-form-urlencoded";
                    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

                    $result = curl_exec($ch);
                    if (curl_errno($ch)) {
                        $get_sms_status = curl_error($ch);
                    }
                    curl_close($ch);

                    $response = json_decode($result, true);

                    if (is_array($response) && array_key_exists('id', $response)) {
                        $get_sms_status = 'Delivered|'.$response['id'];
                    } elseif (is_array($response) && array_key_exists('errors', $response)) {
                        $get_sms_status = $response['errors'][0]['description'];
                    } else {
                        $get_sms_status = 'Unknown Error';
                    }
                    break;

                case 'WhatsAppChatApi':

                    $data = [
                            'phone' => $phone,
                            'body'  => $message,
                    ];
                    $json = json_encode($data);

                    $url     = $gateway_url.'/message?token='.$sending_server->api_token;
                    $options = stream_context_create([
                            'http' => [
                                    'method'  => 'POST',
                                    'header'  => 'Content-type: application/json',
                                    'content' => $json,
                            ],
                    ]);

                    try {
                        $result = file_get_contents($url, false, $options);

                        $json_array[] = [];
                        $json_array   = json_decode($result, true);

                        if (isset($json_array) && is_array($json_array) && array_key_exists('sent', $json_array)) {
                            if ($json_array['sent'] == true) {
                                $get_sms_status = 'Success|'.$json_array['queueNumber'];
                            } else {
                                $get_sms_status = $json_array['message'];
                            }
                        } else {
                            $get_sms_status = 'Invalid request';
                        }

                    } catch (Exception $ex) {
                        $get_sms_status = $ex->getMessage();
                    }

                    break;


                default:
                    $get_sms_status = __('locale.sending_servers.sending_server_not_found');
                    break;
            }
        }


        $reportsData = [
                'user_id'           => $data['user_id'],
                'to'                => $phone,
                'message'           => $message,
                'sms_type'          => 'whatsapp',
                'status'            => $get_sms_status,
                'cost'              => $data['cost'],
                'sending_server_id' => $sending_server->id,
        ];

        if (isset($data['sender_id'])) {
            $reportsData['from'] = $data['sender_id'];
        }

        if (isset($data['campaign_id'])) {
            $reportsData['campaign_id'] = $data['campaign_id'];
        }

        if (isset($data['api_key'])) {
            $reportsData['api_key'] = $data['api_key'];
            $reportsData['send_by'] = 'api';
        } else {
            $reportsData['send_by'] = 'from';
        }

        $status = Reports::create($reportsData);

        if ($status) {
            return $status;
        }

        return __('locale.exceptions.something_went_wrong');

    }
}
